# Video API Backend

Private video hosting API for Android app with admin panel.

## Features

- Upload and manage up to 20 videos
- RESTful API with authentication
- Admin panel for video management
- Secure video streaming
- Support for 16:9 and 9:16 aspect ratios
- SQLite database (lightweight, no MySQL needed)

## File Structure

```
video_upload_api/
├── admin/                      # Admin panel
│   ├── index.php              # Dashboard with upload UI
│   ├── login.php              # Login page
│   ├── logout.php             # Logout handler
│   ├── upload_handler.php     # Upload processing
│   ├── delete_handler.php     # Delete processing
│   └── assets/
│       ├── style.css          # Admin panel styles
│       └── script.js          # Admin panel scripts
│
├── api/                        # API endpoints
│   ├── videos.php             # GET all videos
│   ├── video.php              # GET single video
│   ├── stream.php             # Stream video file
│   └── config.php             # Configuration
│
├── includes/                   # Shared PHP files
│   ├── auth.php               # API authentication
│   ├── admin_auth.php         # Admin authentication
│   ├── db.php                 # Database functions
│   └── helpers.php            # Utility functions
│
├── videos/                     # Video storage (protected)
├── database/                   # SQLite database
├── .htaccess                   # Security rules
└── index.php                   # Root page
```

## Installation

### 1. Upload Files to cPanel

Upload all files to your server (e.g., `/public_html/video-api/`)

### 2. Configure Settings

Edit `api/config.php` and update:

```php
// Change this to a random 32-character string
define('API_KEY', 'YOUR_32_CHARACTER_RANDOM_API_KEY');

// Update with your actual domain
define('BASE_URL', 'https://yourdomain.com/video-api');

// Generate a new password hash
// Run: php -r "echo password_hash('your_password', PASSWORD_BCRYPT);"
define('ADMIN_PASSWORD_HASH', 'generated_hash_here');
```

### 3. Set Permissions

Set the following directory permissions:

```bash
chmod 755 videos/
chmod 755 database/
chmod 644 api/config.php
```

### 4. Test Installation

1. Visit `https://yourdomain.com/video-api/`
2. Click "Admin Login"
3. Login with:
   - Username: `admin`
   - Password: `admin123` (change this immediately!)

### 5. Security Checklist

Before going live:

- [ ] Change admin password in `api/config.php`
- [ ] Generate unique API key
- [ ] Update `BASE_URL` with your actual domain
- [ ] Enable HTTPS/SSL
- [ ] Test all endpoints with Postman
- [ ] Verify videos can't be accessed directly

## API Usage

### Authentication

All API requests require an API key in the header:

```
X-API-Key: your_api_key_here
```

### Endpoints

#### GET /api/videos.php

Get all videos in random order.

**Request:**
```http
GET /video-api/api/videos.php
X-API-Key: your_api_key_here
```

**Response:**
```json
{
  "success": true,
  "count": 15,
  "videos": [
    {
      "id": "vid_a1b2c3d4e5f6",
      "filename": "video1.mp4",
      "aspect_ratio": "16:9",
      "url": "https://yourdomain.com/video-api/api/stream.php?id=vid_a1b2c3d4e5f6",
      "file_size": "15728640",
      "duration": 45,
      "upload_date": "2025-11-25 14:30:00"
    }
  ]
}
```

#### GET /api/video.php?id={video_id}

Get single video details.

**Request:**
```http
GET /video-api/api/video.php?id=vid_a1b2c3d4e5f6
X-API-Key: your_api_key_here
```

#### GET /api/stream.php?id={video_id}

Stream video file (supports range requests for seeking).

**Request:**
```http
GET /video-api/api/stream.php?id=vid_a1b2c3d4e5f6
X-API-Key: your_api_key_here
```

## Admin Panel

### Upload Video

1. Login to admin panel
2. Drag & drop video or click to browse
3. Select aspect ratio (16:9 or 9:16)
4. Click "Upload Video"

### Delete Video

1. Find video in the list
2. Click "Delete" button
3. Confirm deletion

## Android Integration

### Retrofit Interface

```kotlin
interface VideoApiService {
    @GET("api/videos.php")
    fun getVideos(
        @Header("X-API-Key") apiKey: String
    ): Call<VideoResponse>
}
```

### Data Models

```kotlin
data class VideoResponse(
    val success: Boolean,
    val count: Int,
    val videos: List<Video>
)

data class Video(
    val id: String,
    val filename: String,
    val aspect_ratio: String,
    val url: String,
    val file_size: String,
    val duration: Int,
    val upload_date: String
)
```

## Troubleshooting

### Database Errors

If you see database errors, the database will auto-initialize on first API call or admin login.

### Upload Errors

Check PHP upload limits in `php.ini`:
```ini
upload_max_filesize = 100M
post_max_size = 100M
max_execution_time = 300
```

### Video Streaming Issues

Ensure `.htaccess` files are properly configured and Apache `mod_rewrite` is enabled.

### CORS Issues

If Android app can't access API, check CORS headers in `.htaccess`:
```apache
Header set Access-Control-Allow-Origin "*"
Header set Access-Control-Allow-Headers "X-API-Key, Content-Type"
```

## Support

For issues or questions:
1. Check server error logs
2. Enable `DEBUG_MODE` in `api/config.php`
3. Check browser console for JavaScript errors

## License

Private use only.
