<?php
require_once __DIR__ . '/../includes/admin_auth.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/helpers.php';

// Require authentication
requireAdminAuth();

// Get video count and list
$videoCount = getVideoCount();
$videos = getAllVideos(false); // Don't randomize for admin panel
$maxVideos = MAX_VIDEOS;
$canUpload = $videoCount < $maxVideos;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Video Management - Admin Panel</title>
    <link rel="stylesheet" href="assets/style.css">
</head>
<body>
    <div class="container">
        <header class="header">
            <h1>Video Management Panel</h1>
            <div class="header-actions">
                <span class="video-counter">Videos: <?= $videoCount ?> / <?= $maxVideos ?></span>
                <a href="logout.php" class="btn btn-secondary">Logout</a>
            </div>
        </header>

        <?php if ($canUpload): ?>
        <section class="upload-section">
            <h2>Upload New Video</h2>
            <form id="uploadForm" enctype="multipart/form-data">
                <div class="upload-area" id="uploadArea">
                    <input type="file" id="videoFile" name="video" accept="video/mp4,video/webm,video/quicktime" hidden>
                    <div class="upload-content">
                        <svg width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path>
                            <polyline points="17 8 12 3 7 8"></polyline>
                            <line x1="12" y1="3" x2="12" y2="15"></line>
                        </svg>
                        <p>Drag & drop video here or click to browse</p>
                        <p class="upload-hint">MP4, WebM, MOV (Max 100MB)</p>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label for="aspectRatio">Aspect Ratio</label>
                        <select id="aspectRatio" name="aspect_ratio" required>
                            <option value="16:9">16:9 (Landscape)</option>
                            <option value="9:16">9:16 (Portrait/Vertical)</option>
                        </select>
                    </div>
                </div>

                <div class="progress-container" id="progressContainer" style="display: none;">
                    <div class="progress-bar">
                        <div class="progress-fill" id="progressFill"></div>
                    </div>
                    <p class="progress-text" id="progressText">Uploading...</p>
                </div>

                <div class="alert" id="uploadAlert" style="display: none;"></div>

                <button type="submit" class="btn btn-primary" id="uploadBtn">Upload Video</button>
            </form>
        </section>
        <?php else: ?>
        <div class="alert alert-warning">
            Maximum video limit reached (<?= $maxVideos ?>). Delete videos to upload new ones.
        </div>
        <?php endif; ?>

        <section class="videos-section">
            <h2>Uploaded Videos</h2>

            <?php if (empty($videos)): ?>
            <div class="empty-state">
                <p>No videos uploaded yet.</p>
            </div>
            <?php else: ?>
            <div class="videos-grid">
                <?php foreach ($videos as $video): ?>
                <div class="video-card" data-video-id="<?= htmlspecialchars($video['video_id']) ?>">
                    <div class="video-preview">
                        <video src="stream.php?id=<?= htmlspecialchars($video['video_id']) ?>" preload="metadata" controls></video>
                        <div class="video-overlay">
                            <span class="aspect-badge"><?= htmlspecialchars($video['aspect_ratio']) ?></span>
                        </div>
                    </div>
                    <div class="video-info">
                        <h3><?= htmlspecialchars($video['original_filename']) ?></h3>
                        <p class="video-meta">
                            <span>Size: <?= formatFileSize($video['file_size']) ?></span>
                            <?php if ($video['duration'] > 0): ?>
                            <span>Duration: <?= gmdate("i:s", $video['duration']) ?></span>
                            <?php endif; ?>
                        </p>
                        <p class="video-date">Uploaded: <?= date('M j, Y g:i A', strtotime($video['upload_date'])) ?></p>
                        <div class="video-actions">
                            <button class="btn btn-small btn-danger" onclick="deleteVideo('<?= htmlspecialchars($video['video_id']) ?>')">Delete</button>
                            <a href="stream.php?id=<?= htmlspecialchars($video['video_id']) ?>" class="btn btn-small btn-secondary" target="_blank">View</a>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </section>
    </div>

    <script src="assets/script.js"></script>
</body>
</html>
