<?php
require_once __DIR__ . '/../includes/admin_auth.php';
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/../includes/helpers.php';

// Require authentication
if (!isAdminLoggedIn()) {
    sendErrorResponse('Unauthorized', 401);
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    sendErrorResponse('Invalid request method', 405);
}

// Check if video file is uploaded
if (!isset($_FILES['video']) || $_FILES['video']['error'] !== UPLOAD_ERR_OK) {
    $errorMessages = [
        UPLOAD_ERR_INI_SIZE => 'File exceeds upload_max_filesize',
        UPLOAD_ERR_FORM_SIZE => 'File exceeds MAX_FILE_SIZE',
        UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
        UPLOAD_ERR_NO_FILE => 'No file was uploaded',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
        UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload'
    ];

    $error = isset($_FILES['video']) ? $errorMessages[$_FILES['video']['error']] ?? 'Unknown upload error' : 'No file uploaded';
    sendErrorResponse($error);
}

$videoFile = $_FILES['video'];

// Validate video file
$validation = validateVideoFile($videoFile);
if (!$validation['valid']) {
    sendErrorResponse($validation['error']);
}

// Check video count limit
$currentCount = getVideoCount();
if ($currentCount >= MAX_VIDEOS) {
    sendErrorResponse('Maximum video limit reached (' . MAX_VIDEOS . ')');
}

// Get aspect ratio
$aspectRatio = $_POST['aspect_ratio'] ?? 'auto';
if ($aspectRatio === 'auto') {
    $aspectRatio = detectAspectRatio($videoFile['name']);
} elseif (!in_array($aspectRatio, ['16:9', '9:16'])) {
    $aspectRatio = '16:9';
}

try {
    // Generate unique video ID
    $videoId = generateVideoId();

    // Sanitize and create unique filename
    $originalFilename = $videoFile['name'];
    $extension = getFileExtension($originalFilename);
    $sanitizedName = sanitizeFilename($originalFilename);
    $filename = pathinfo($sanitizedName, PATHINFO_FILENAME) . '_' . substr($videoId, 4, 8) . '.' . $extension;

    // Ensure upload directory exists
    if (!file_exists(UPLOAD_DIR)) {
        mkdir(UPLOAD_DIR, 0755, true);
    }

    // Move uploaded file
    $targetPath = UPLOAD_DIR . $filename;
    if (!move_uploaded_file($videoFile['tmp_name'], $targetPath)) {
        throw new Exception('Failed to move uploaded file');
    }

    // Get video duration (optional, requires FFmpeg)
    $duration = getVideoDuration($targetPath);

    // Build video URL
    $videoUrl = STREAM_URL . '?id=' . $videoId;

    // Create database record
    $videoData = [
        'video_id' => $videoId,
        'filename' => $filename,
        'original_filename' => $originalFilename,
        'aspect_ratio' => $aspectRatio,
        'file_size' => $videoFile['size'],
        'duration' => $duration,
        'url' => $videoUrl
    ];

    if (!createVideo($videoData)) {
        // Delete file if database insert fails
        unlink($targetPath);
        throw new Exception('Failed to create database record');
    }

    // Log the upload
    logMessage("Video uploaded: $videoId ($originalFilename) by admin", 'info');

    // Return success response
    sendJSONResponse([
        'success' => true,
        'message' => 'Video uploaded successfully',
        'video' => [
            'id' => $videoId,
            'filename' => $filename,
            'original_filename' => $originalFilename,
            'aspect_ratio' => $aspectRatio,
            'file_size' => formatFileSize($videoFile['size']),
            'duration' => $duration,
            'url' => $videoUrl
        ]
    ]);

} catch (Exception $e) {
    error_log("Upload error: " . $e->getMessage());
    sendErrorResponse('Upload failed: ' . $e->getMessage(), 500);
}
?>
