<?php
// API Authentication Middleware

require_once __DIR__ . '/../api/config.php';

/**
 * Validate API key from request headers
 * @return bool True if valid, false otherwise
 */
function validateAPIKey() {
    // Check X-API-Key header
    $headers = getallheaders();

    if (!$headers) {
        return false;
    }

    // Try different header formats (case-insensitive)
    $apiKey = null;
    foreach ($headers as $key => $value) {
        if (strtolower($key) === 'x-api-key') {
            $apiKey = $value;
            break;
        }
    }

    // Also check for Authorization header (Bearer token format)
    if (!$apiKey && isset($headers['Authorization'])) {
        $auth = $headers['Authorization'];
        if (preg_match('/Bearer\s+(.*)$/i', $auth, $matches)) {
            $apiKey = $matches[1];
        }
    }

    return $apiKey && $apiKey === API_KEY;
}

/**
 * Send unauthorized response and exit
 * @param string $message Error message
 */
function sendUnauthorizedResponse($message = 'Invalid API key') {
    http_response_code(401);
    header('Content-Type: application/json');
    echo json_encode([
        'success' => false,
        'error' => $message
    ]);
    exit;
}

/**
 * Require valid API key (middleware function)
 * Call this at the beginning of any API endpoint
 */
function requireAPIKey() {
    if (!validateAPIKey()) {
        sendUnauthorizedResponse();
    }
}

/**
 * Get all request headers (polyfill for getallheaders)
 * @return array Headers
 */
if (!function_exists('getallheaders')) {
    function getallheaders() {
        $headers = [];
        foreach ($_SERVER as $name => $value) {
            if (substr($name, 0, 5) == 'HTTP_') {
                $headers[str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))))] = $value;
            }
        }
        return $headers;
    }
}
?>
