<?php
// Database Functions for Video API

require_once __DIR__ . '/../api/config.php';

/**
 * Get database connection
 * @return PDO Database connection
 */
function getDBConnection() {
    try {
        // Check if database file exists, if not initialize it
        if (!file_exists(DB_PATH)) {
            initializeDatabase();
        }

        $db = new PDO('sqlite:' . DB_PATH);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $db->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        return $db;
    } catch (PDOException $e) {
        error_log("Database connection failed: " . $e->getMessage());
        throw new Exception("Database connection failed");
    }
}

/**
 * Initialize database with schema
 */
function initializeDatabase() {
    try {
        // Create database directory if it doesn't exist
        $dbDir = dirname(DB_PATH);
        if (!file_exists($dbDir)) {
            mkdir($dbDir, 0755, true);
        }

        $db = new PDO('sqlite:' . DB_PATH);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Create videos table
        $createTableSQL = "
        CREATE TABLE IF NOT EXISTS videos (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            video_id VARCHAR(50) UNIQUE NOT NULL,
            filename VARCHAR(255) NOT NULL,
            original_filename VARCHAR(255) NOT NULL,
            aspect_ratio VARCHAR(10) NOT NULL,
            file_size INTEGER NOT NULL,
            duration INTEGER DEFAULT 0,
            url TEXT NOT NULL,
            upload_date DATETIME DEFAULT CURRENT_TIMESTAMP,
            status VARCHAR(20) DEFAULT 'active'
        );
        ";

        $db->exec($createTableSQL);

        // Create indexes
        $db->exec("CREATE INDEX IF NOT EXISTS idx_video_id ON videos(video_id);");
        $db->exec("CREATE INDEX IF NOT EXISTS idx_aspect_ratio ON videos(aspect_ratio);");

        error_log("Database initialized successfully");
    } catch (PDOException $e) {
        error_log("Database initialization failed: " . $e->getMessage());
        throw new Exception("Database initialization failed");
    }
}

/**
 * Create new video record
 * @param array $data Video data
 * @return bool Success status
 */
function createVideo($data) {
    try {
        $db = getDBConnection();

        $stmt = $db->prepare("
            INSERT INTO videos (video_id, filename, original_filename, aspect_ratio, file_size, duration, url, status)
            VALUES (:video_id, :filename, :original_filename, :aspect_ratio, :file_size, :duration, :url, 'active')
        ");

        return $stmt->execute([
            ':video_id' => $data['video_id'],
            ':filename' => $data['filename'],
            ':original_filename' => $data['original_filename'],
            ':aspect_ratio' => $data['aspect_ratio'],
            ':file_size' => $data['file_size'],
            ':duration' => $data['duration'] ?? 0,
            ':url' => $data['url']
        ]);
    } catch (PDOException $e) {
        error_log("Error creating video: " . $e->getMessage());
        return false;
    }
}

/**
 * Get all active videos
 * @param bool $randomize Whether to randomize the order
 * @return array Array of videos
 */
function getAllVideos($randomize = true) {
    try {
        $db = getDBConnection();

        $orderBy = $randomize ? "ORDER BY RANDOM()" : "ORDER BY upload_date DESC";
        $stmt = $db->query("SELECT * FROM videos WHERE status = 'active' " . $orderBy);

        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Error fetching videos: " . $e->getMessage());
        return [];
    }
}

/**
 * Get video by ID
 * @param string $videoId Video ID
 * @return array|null Video data or null
 */
function getVideoById($videoId) {
    try {
        $db = getDBConnection();

        $stmt = $db->prepare("SELECT * FROM videos WHERE video_id = :video_id AND status = 'active'");
        $stmt->execute([':video_id' => $videoId]);

        $result = $stmt->fetch();
        return $result ? $result : null;
    } catch (PDOException $e) {
        error_log("Error fetching video: " . $e->getMessage());
        return null;
    }
}

/**
 * Get video by filename
 * @param string $filename Filename
 * @return array|null Video data or null
 */
function getVideoByFilename($filename) {
    try {
        $db = getDBConnection();

        $stmt = $db->prepare("SELECT * FROM videos WHERE filename = :filename");
        $stmt->execute([':filename' => $filename]);

        $result = $stmt->fetch();
        return $result ? $result : null;
    } catch (PDOException $e) {
        error_log("Error fetching video by filename: " . $e->getMessage());
        return null;
    }
}

/**
 * Delete video (soft delete - mark as deleted)
 * @param string $videoId Video ID
 * @return bool Success status
 */
function deleteVideo($videoId) {
    try {
        $db = getDBConnection();

        $stmt = $db->prepare("UPDATE videos SET status = 'deleted' WHERE video_id = :video_id");
        return $stmt->execute([':video_id' => $videoId]);
    } catch (PDOException $e) {
        error_log("Error deleting video: " . $e->getMessage());
        return false;
    }
}

/**
 * Permanently delete video from database
 * @param string $videoId Video ID
 * @return bool Success status
 */
function permanentlyDeleteVideo($videoId) {
    try {
        $db = getDBConnection();

        $stmt = $db->prepare("DELETE FROM videos WHERE video_id = :video_id");
        return $stmt->execute([':video_id' => $videoId]);
    } catch (PDOException $e) {
        error_log("Error permanently deleting video: " . $e->getMessage());
        return false;
    }
}

/**
 * Get total count of active videos
 * @return int Video count
 */
function getVideoCount() {
    try {
        $db = getDBConnection();

        $stmt = $db->query("SELECT COUNT(*) as count FROM videos WHERE status = 'active'");
        $result = $stmt->fetch();

        return $result['count'];
    } catch (PDOException $e) {
        error_log("Error counting videos: " . $e->getMessage());
        return 0;
    }
}

/**
 * Update video metadata
 * @param string $videoId Video ID
 * @param array $data Data to update
 * @return bool Success status
 */
function updateVideo($videoId, $data) {
    try {
        $db = getDBConnection();

        $allowedFields = ['filename', 'original_filename', 'aspect_ratio', 'file_size', 'duration', 'url', 'status'];
        $updates = [];
        $params = [':video_id' => $videoId];

        foreach ($data as $key => $value) {
            if (in_array($key, $allowedFields)) {
                $updates[] = "$key = :$key";
                $params[":$key"] = $value;
            }
        }

        if (empty($updates)) {
            return false;
        }

        $sql = "UPDATE videos SET " . implode(', ', $updates) . " WHERE video_id = :video_id";
        $stmt = $db->prepare($sql);

        return $stmt->execute($params);
    } catch (PDOException $e) {
        error_log("Error updating video: " . $e->getMessage());
        return false;
    }
}
?>
