<?php
// Helper Utility Functions

require_once __DIR__ . '/../api/config.php';

/**
 * Generate unique video ID
 * @return string Unique video ID
 */
function generateVideoId() {
    return 'vid_' . bin2hex(random_bytes(12));
}

/**
 * Get file extension
 * @param string $filename Filename
 * @return string Extension
 */
function getFileExtension($filename) {
    return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
}

/**
 * Format file size
 * @param int $bytes File size in bytes
 * @return string Formatted file size
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}

/**
 * Validate video file
 * @param array $file $_FILES array element
 * @return array ['valid' => bool, 'error' => string]
 */
function validateVideoFile($file) {
    // Check if file was uploaded
    if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
        return ['valid' => false, 'error' => 'No file uploaded'];
    }

    // Check file size
    if ($file['size'] > MAX_FILE_SIZE) {
        return ['valid' => false, 'error' => 'File size exceeds maximum allowed (' . formatFileSize(MAX_FILE_SIZE) . ')'];
    }

    // Check file extension
    $extension = getFileExtension($file['name']);
    if (!in_array($extension, ALLOWED_EXTENSIONS)) {
        return ['valid' => false, 'error' => 'Invalid file extension. Allowed: ' . implode(', ', ALLOWED_EXTENSIONS)];
    }

    // Check MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);

    if (!in_array($mimeType, ALLOWED_TYPES)) {
        return ['valid' => false, 'error' => 'Invalid file type'];
    }

    return ['valid' => true];
}

/**
 * Detect video aspect ratio from filename or metadata
 * @param string $filename Filename
 * @return string Aspect ratio (16:9 or 9:16)
 */
function detectAspectRatio($filename) {
    // Check filename for hints
    $filename = strtolower($filename);

    if (strpos($filename, '9-16') !== false || strpos($filename, '9x16') !== false ||
        strpos($filename, 'vertical') !== false || strpos($filename, 'portrait') !== false ||
        strpos($filename, 'story') !== false || strpos($filename, 'short') !== false) {
        return '9:16';
    }

    // Default to 16:9 (landscape)
    return '16:9';
}

/**
 * Get video duration (requires FFmpeg - optional)
 * @param string $filePath Path to video file
 * @return int Duration in seconds (0 if unable to detect)
 */
function getVideoDuration($filePath) {
    // Check if FFmpeg/FFprobe is available
    $ffprobePath = 'ffprobe'; // Or full path: /usr/bin/ffprobe

    if (!command_exists($ffprobePath)) {
        return 0;
    }

    try {
        $command = escapeshellcmd($ffprobePath) . ' -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 ' . escapeshellarg($filePath);
        $output = shell_exec($command);

        if ($output) {
            return (int)round(floatval(trim($output)));
        }
    } catch (Exception $e) {
        error_log("Error getting video duration: " . $e->getMessage());
    }

    return 0;
}

/**
 * Check if command exists
 * @param string $command Command name
 * @return bool True if exists
 */
function command_exists($command) {
    $whereIsCommand = PHP_OS_FAMILY === 'Windows' ? 'where' : 'which';
    $result = shell_exec("$whereIsCommand $command 2>&1");
    return !empty($result);
}

/**
 * Sanitize filename
 * @param string $filename Original filename
 * @return string Sanitized filename
 */
function sanitizeFilename($filename) {
    // Remove any path components
    $filename = basename($filename);

    // Get extension
    $extension = getFileExtension($filename);
    $nameWithoutExt = pathinfo($filename, PATHINFO_FILENAME);

    // Remove special characters, keep alphanumeric, dash, underscore
    $nameWithoutExt = preg_replace('/[^a-zA-Z0-9_-]/', '_', $nameWithoutExt);

    // Limit length
    $nameWithoutExt = substr($nameWithoutExt, 0, 50);

    return $nameWithoutExt . '.' . $extension;
}

/**
 * Send JSON response
 * @param array $data Response data
 * @param int $statusCode HTTP status code
 */
function sendJSONResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data);
    exit;
}

/**
 * Send error response
 * @param string $message Error message
 * @param int $statusCode HTTP status code
 */
function sendErrorResponse($message, $statusCode = 400) {
    sendJSONResponse([
        'success' => false,
        'error' => $message
    ], $statusCode);
}

/**
 * Sanitize output (prevent XSS)
 * @param string $string Input string
 * @return string Sanitized string
 */
function sanitizeOutput($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Log message to file
 * @param string $message Log message
 * @param string $level Log level (info, error, warning)
 */
function logMessage($message, $level = 'info') {
    if (!DEBUG_MODE) {
        return;
    }

    $logFile = __DIR__ . '/../logs/app.log';
    $logDir = dirname($logFile);

    if (!file_exists($logDir)) {
        mkdir($logDir, 0755, true);
    }

    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message" . PHP_EOL;

    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

/**
 * Get client IP address
 * @return string IP address
 */
function getClientIP() {
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        return $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        return $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
    }
}
?>
